#include "searchfacade.h"

#include <iostream>
#include <cassert>

//////////////////////////////////////////////////////////////////////////

SearchFacadePtr GetSearchFacadeInstance()
{
	static SearchFacadePtr sSearchFacade;
	if (!sSearchFacade)
	{
		sSearchFacade = SearchFacadePtr(new SearchFacadeImpl);
	}
	return sSearchFacade;
}

//////////////////////////////////////////////////////////////////////////

const std::string kDbPath = "D:/repos/testxapian_minimal/xapiandb.glass";

bool SearchFacadeImpl::Init()
{
	try
	{
		if (!mWritableDataBase)
		{
			mWritableDataBase = XpWritableDataBasePtr(new Xapian::WritableDatabase(
				kDbPath, Xapian::DB_CREATE_OR_OPEN));
		}

	}
	catch (const Xapian::Error &ex)
	{
		std::cout << ex.get_msg() << std::endl;

		return false;
	}

	mRangeField.clear();

	int slot = 1;
	for (const auto& field : mRangeFieldName)
	{
		mRangeField[field] = slot++;
	}

	return true;
}

bool SearchFacadeImpl::UnInit()
{
	mRangeField.clear();
	return true;
}

bool SearchFacadeImpl::ConstructQuery(
	const RelativeCond& cond, 
	const RangeCond& rangeCond, 
	Xapian::Query& query)
{
	std::cout << "ConstructQuery begin" << std::endl;
	bool ret = true;

	std::shared_lock<std::shared_mutex> readLock(mMutex);
	std::cout << "ConstructQuery readLock" << std::endl;

	if (!mWritableDataBase)
	{
		std::cout << "Null of Point to DB" << std::endl;
		return false;
	}

	try
	{
		//Construct Query object tree
		Xapian::QueryParser queryParser;
		Xapian::Database db(kDbPath);
		queryParser.set_database(db);

		PrepareQueryParser(queryParser);

		ret = ConstructQueryObjectTree(cond, rangeCond, queryParser, query, "");
		assert(ret == true);
		if (!ret)
		{
			std::cout << "ConstructQuery fail, ConstructQueryObjectTree error" << std::endl;
			return ret;
		}
	}
	catch (const Xapian::Error &ex)
	{
		std::cout << ex.get_msg() << std::endl;
	}

	std::cout << "ConstructQuery end" << std::endl;
	return ret;
}

bool SearchFacadeImpl::QueryBusinessID(
	const Xapian::Query& query, 
	const int& offSet, const int& pageSize, 
	std::vector<int>& businessId, 
	const std::string& fillCreator /*= ""*/)
{
	std::cout << "QueryBusinessID by Query begin" << std::endl;

	bool ret = true;

	std::shared_lock<std::shared_mutex> readLock(mMutex);
	std::cout << "QueryBusinessID by Query readLock" << std::endl;

	if (!mWritableDataBase)
	{
		std::cout << "Null of Point to DB" << std::endl;
		return false;
	}

	try
	{
		Xapian::Database db(kDbPath);

		// Use an Enquire object on the database to run the query.
		Xapian::Enquire enquire(db);
		enquire.set_query(query);

		// get result
		Xapian::MSet mset = enquire.get_mset(offSet, pageSize + 1);

		for (Xapian::MSetIterator m = mset.begin(); m != mset.end(); ++m)
		{
			const std::string &data = m.get_document().get_data();
			businessId.push_back(std::stoi(data));
		}
	}
	catch (const Xapian::Error &ex)
	{
		std::cout << ex.get_msg() << std::endl;
	}

	std::cout << "QueryBusinessID by Query end" << std::endl;
	return ret;
}

bool SearchFacadeImpl::ConstructQueryObjectTree(
	const RelativeCond& cond, 
	const RangeCond& rangeCond, 
	Xapian::QueryParser& queryParser, 
	Xapian::Query& query, 
	const std::string& fillCreator /*= ""*/)
{
	std::cout << "ConstructQueryObjectTree begin" << std::endl;

	// ǷΧ
	RelativeCond::const_iterator iteCond = cond.begin();
	for (; iteCond != cond.end(); ++iteCond)
	{
		Xapian::Query andQuery;
		std::string queryString;
		const std::string& field = iteCond->first;

		std::vector<std::string>::const_iterator iteStr = iteCond->second.begin();
		for (; iteStr != iteCond->second.end(); ++iteStr)
		{
			const std::string& condString = *iteStr;
		
			queryString = "\"" + *iteStr + "\"";

			Xapian::Query subQuery;
			subQuery = queryParser.parse_query(queryString,
				Xapian::QueryParser::FLAG_DEFAULT | Xapian::QueryParser::FLAG_WILDCARD,
				iteCond->first);

			if (iteStr == iteCond->second.begin())
			{
				andQuery = subQuery;
			}
			else
			{
				andQuery = Xapian::Query(Xapian::Query::OP_OR, andQuery, subQuery);
			}
		}
		if (iteCond == cond.begin())
		{
			query = andQuery;
		}
		else
		{
			query = Xapian::Query(Xapian::Query::OP_AND, query, andQuery);
		}
	}

	// Χ
	std::string rangeString;
	int count = 1;
	RangeCond::const_iterator iterRangeCond;
	for (iterRangeCond = rangeCond.begin();
		iterRangeCond != rangeCond.end();
		++iterRangeCond, ++count)
	{
		const RangeFieldValue &rangeFieldValue = iterRangeCond->second;
		if (!rangeFieldValue.start.empty())
		{
			rangeString += rangeFieldValue.start + iterRangeCond->first;
			rangeString += "..";
		}
		else
		{
			rangeString += "..";
		}

		if (!rangeFieldValue.end.empty())
		{
			rangeString += rangeFieldValue.end + iterRangeCond->first;
		}
		else
		{
			rangeString += iterRangeCond->first;
		}

		if (count < (int)rangeCond.size())
		{
			rangeString += " AND ";
		}
	}
	if (!rangeString.empty())
	{
		Xapian::Query rangeQuery = queryParser.parse_query(rangeString);
		if (query.empty())
		{
			query = rangeQuery;
			std::cout << "ConstructQueryObjectTree end" << std::endl;
			return false;
		}

		query = Xapian::Query(Xapian::Query::OP_AND, query, rangeQuery);
	}

	std::cout << "ConstructQueryObjectTree end" << std::endl;
	return true;
}

void SearchFacadeImpl::PrepareQueryParser(Xapian::QueryParser &queryParser)
{
	std::map<std::string, int>::const_iterator iteRange = mRangeField.begin();
	for (; iteRange != mRangeField.end(); ++iteRange)
	{
		queryParser.add_prefix(iteRange->first, iteRange->first);

		Xapian::NumberValueRangeProcessor *vrproc =
			new Xapian::NumberValueRangeProcessor(iteRange->second, iteRange->first, false);
		queryParser.add_valuerangeprocessor(vrproc);
	}
}
