#include "unmtest.h"

#include <shared_mutex>
#include <thread>
#include <functional>
#include <typeinfo>
#include <cassert>

#include "windows.h"
#include "safecout.h"

//////////////////////////////////////////////////////////////////////////

typedef std::shared_ptr<Xapian::WritableDatabase> XPWDataBasePtr;
typedef std::shared_ptr<Xapian::NumberValueRangeProcessor> RangeValueProcessorPtr;

struct RangeFieldValue
{
	std::string strStart;
	std::string strEnd;
};

struct XapianData 
{
	void Clear()
	{
		busiId = 0;
		fieldValue.clear();
		rangeFieldValue.clear();
	}

	int busiId;
	std::map<std::string, std::string> fieldValue;
	std::map<std::string, std::string> rangeFieldValue;
};

//////////////////////////////////////////////////////////////////////////

#ifdef XAPIAN_1_4_25
const std::string kDbPath = "D:/repos/testxapian/xapiandb.glass";
#else
const std::string kDbPath = "D:/repos/testxapian/xapiandb.chert";
#endif

const std::string kPassPort = "PP";
const std::string kLinkPort = "TP";

const std::string kDataPath = "D:/repos/testxapian/data.txt";

//////////////////////////////////////////////////////////////////////////

XPWDataBasePtr gWDataBase = nullptr;

std::vector<std::string> gFieldNameVec = {
	"SN", "SB", "SP", "DN", "DB", "DP", "PN", "PB", "PP", "NT", "BRS", "BT", "AS", "CR", "NA",
	"UL", "CN", "ID", "SWC", "DWC", "PWC", "DR", "PT", "SPT", "IEOC", "BW", "SL", "SID", "OVPNID",
	"TB", "ON", "TID", "SLN", "SLB", "SLP", "II", "EI", "TI", "SLL", "DLL", "PLL", "SVI", "DVI",
	"PVI", "IBC", "BPN", "IFIP", "SPC", "BSN", "PWT", "SVCI", "VTT", "NNIT", "BPT", "NTS", "VSI",
	"VLN", "VLI", "VWN", "VWI", "VRIP", "SPR", "DPR", "IPEB", "SSDNE", "SSLP", "NL", "NCK", "SLPR",
	"SLRT", "SLRL", "SLLS", "SLRS", "PCA"
};

std::vector<std::string> gRangeFieldNameVec = {
	"CT", "MT", "LR"
};

//ܰĵ,ҪִʴԶ﷽ʽѯ
std::set<std::string> gIndexField = {
	"NA", "UL", "US101", "US104", "US107", "US108", "US110", "US1001", "US1002"
};

//ֽ֧βͨ,ѯĩβ"*"
std::set<std::string> gWildcardFiled = {
	"CR"
};

//ַ֧Χ(ʱ䡢޸ʱ),valueΪslotͲѯ
std::map<std::string, int> gRangeField;

//ϵѯ(ҵѯĬΪϵѯ)
std::set<std::string> gAddField = {
	kLinkPort
};

// ڹʱύ,Ч
unsigned int gBatchCount = 0;

std::shared_mutex gMutex;

enum TestType
{
	kTestTypeNone = 0x00,
	kTestTypeRead = 0x01,
	kTestTypeWrite = 0x02
};

//////////////////////////////////////////////////////////////////////////
// thread data
std::vector<std::pair<std::string, bool>> gSortRangeField;

std::map<std::string, std::vector<std::string>> gRelatedCondMap;
std::map<std::string, std::vector<std::string>> gNonRelatedCondMap;
std::map<std::string, RangeFieldValue> gRangeCondMap;
Xapian::Query gRelatedQuery;
Xapian::Query gNonRelatedQuery;

std::vector<XapianData> gData;

// SafeCout safeCout;
// #define SAFE_COUT safeCout
#define SAFE_COUT std::cout

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void PrepareQueryParser(Xapian::QueryParser &oQueryParser)
{
	std::map<std::string, int>::const_iterator iteRange = gRangeField.begin();
	for (; iteRange != gRangeField.end(); ++iteRange)
	{
		oQueryParser.add_prefix(iteRange->first, iteRange->first);

		Xapian::NumberValueRangeProcessor *vrproc =
			new Xapian::NumberValueRangeProcessor(iteRange->second, iteRange->first, false);
		oQueryParser.add_valuerangeprocessor(vrproc);
	}
}

std::wstring ToUnicodeString(const char* pstr)
{
	int iLen = ::MultiByteToWideChar(
		936, NULL, pstr, strlen(pstr), NULL, 0);
	wchar_t* wszString = new wchar_t[iLen + 1];
	std::wstring wstrResult;
	if (NULL == wszString)
	{
		SAFE_COUT << "ToUnicodeString Cannot alloc Memory!" << std::endl;
		return wstrResult;
	}
	::MultiByteToWideChar(936, NULL, pstr, strlen(pstr), wszString, iLen);
	wszString[iLen] = '\0';
	wstrResult = std::wstring(wszString);
	delete[] wszString;
	wszString = NULL;
	return wstrResult;
}

std::wstring ToUnicodeString(const std::string& str)
{
	if (str.empty())
	{
		return L"";
	}

	return ToUnicodeString(str.c_str());
}

std::string UnicodeToUTF8String(const wchar_t* pwstrFrom)
{
	int iLen = ::WideCharToMultiByte(
		CP_UTF8, NULL, pwstrFrom, wcslen(pwstrFrom), NULL, 0, NULL, NULL);
	char* szString = new char[iLen + 1];
	std::string strBuffer;
	if (NULL == szString)
	{
		SAFE_COUT << "UnicodeToUTF8String Cannot alloc Memory!" << std::endl;
		return strBuffer;
	}
	::WideCharToMultiByte(
		CP_UTF8, NULL, pwstrFrom, wcslen(pwstrFrom), szString, iLen, NULL, NULL);
	szString[iLen] = '\0';
	strBuffer = std::string(szString);
	delete[] szString;
	szString = NULL;
	return strBuffer;
}

std::string UnicodeToUTF8String(const std::wstring& wstrFrom)
{
	if (wstrFrom.empty())
	{
		return std::string("");
	}
	return UnicodeToUTF8String(wstrFrom.c_str());
}

std::string ConvertQueryCond(const std::string &strWords)
{
	// ASCַ
	std::wstring wstrWords(L"");
	wstrWords = ToUnicodeString(strWords);
	if (wstrWords.empty())
	{
		return "";
	}

	size_t iSize = wstrWords.length();
	std::wstring wstrConvWords(L"");
	wstrConvWords.reserve(2 * iSize);
	for (size_t i = 0; i < iSize; ++i)
	{
		wchar_t &wChar = wstrWords[i];
		if (wChar != 0)
		{
			if (wChar > 0x00FF)
			{
				if (!wstrConvWords.empty())
				{
					std::wstring wstrTmp = wstrConvWords.substr(wstrConvWords.length() - 1, 1);
					if ((wstrTmp != L" ") && (wstrTmp != L"\""))
					{
						wstrConvWords.append(L" ", 1);
					}
				}

				wstrConvWords.push_back(wChar);
				wstrConvWords.append(L" ", 1);
			}
			else
			{
				wstrConvWords.push_back(wChar);
				wstrConvWords.append(L" ", 1);
			}
		}
	}

	if (!wstrConvWords.empty())
	{
		std::wstring wstrTmp = wstrConvWords.substr(wstrConvWords.length() - 1, 1);
		if (wstrTmp == L" ")
		{
			wstrConvWords.erase(wstrConvWords.length() - 1);
		}
	}

	std::string strConvWords = UnicodeToUTF8String(wstrConvWords);
	return strConvWords;
}

//
void DealFillCreateCondition(const std::string& strFiled,
	const std::string& strFillCreator,
	Xapian::QueryParser& oQueryParser,
	Xapian::Query &oQuery)
{
	SAFE_COUT << "DealFillCreateCondition begin" << std::endl;
	if (strFiled == "CN" && !strFillCreator.empty())
	{
		std::string strCreatorQuery = strFillCreator;
		strCreatorQuery += "*";

		Xapian::Query subTempQuery = oQueryParser.parse_query(strCreatorQuery,
			Xapian::QueryParser::FLAG_DEFAULT | Xapian::QueryParser::FLAG_WILDCARD,
			"CR");

		oQuery = Xapian::Query(Xapian::Query::OP_OR, oQuery, subTempQuery);
	}
	SAFE_COUT << "DealFillCreateCondition end" << std::endl;
}

bool ConstructQueryObjectTree(
	const std::map<std::string, std::vector<std::string> > &oCondMap,
	const std::map<std::string, RangeFieldValue> &oRangeCondMap,
	Xapian::QueryParser &oQueryParser,
	Xapian::Query &oQuery,
	const std::string &strFillCreator)
{
	SAFE_COUT << "ConstructQueryObjectTree begin" << std::endl;

	// ǷΧ
	std::map<std::string, std::vector<std::string> >::const_iterator iteCond = oCondMap.begin();
	for (; iteCond != oCondMap.end(); ++iteCond)
	{
		Xapian::Query andQuery;
		std::string strQuery;
		const std::string &strFiled = iteCond->first;

		// Ƿ֧,Ҫִʴ
		bool bNeedIndex = false;
		std::set<std::string>::const_iterator iteIndex = gIndexField.begin();
		for (; iteIndex != gIndexField.end(); ++iteIndex)
		{
			const std::string &strIte = *iteIndex;
			if (strFiled.substr(0, strIte.size()) == strIte)
			{
				bNeedIndex = true;
				break;
			}
		}

		// ǷҪӽβͨ
		bool bNeedWild = false;
		std::set<std::string>::const_iterator iteWildcard = gWildcardFiled.begin();
		for (; iteWildcard != gWildcardFiled.end(); ++iteWildcard)
		{
			const std::string &strIte = *iteWildcard;
			if (strFiled.substr(0, strIte.size()) == strIte)
			{
				bNeedWild = true;
				break;
			}
		}

		// Ƿϵѯ
		bool bAddFiled = false;
		std::set<std::string>::const_iterator iteAddFiled = gAddField.begin();
		for (; iteAddFiled != gAddField.end(); ++iteAddFiled)
		{
			const std::string &strIte = *iteAddFiled;
			if (strFiled.substr(0, strIte.size()) == strIte)
			{
				bAddFiled = true;
				break;
			}
		}

		std::vector<std::string>::const_iterator iteStr = iteCond->second.begin();
		for (; iteStr != iteCond->second.end(); ++iteStr)
		{
			const std::string &strCond = *iteStr;
			// ת,ĵִַ
			if (bNeedIndex)
			{
				if (!strCond.empty())
				{
					strQuery = ConvertQueryCond(strCond);
					if (strQuery.empty())
					{
						SAFE_COUT << "convert query string failed, str = " << strCond << std::endl;
						SAFE_COUT << "ConstructQueryObjectTree fail, ConvertQueryCond error" << std::endl;
						return false;
					}
				}
			}
			else
			{
				strQuery = *iteStr;
				//澯Էѯҵͨҵ߼5(RPC_QUERYBUSSINESS_TYPE_ALARMRELATION_ALL_BUSINESS)Ϊ澯Էҵѯͣ1Ϊͨҵ
				if (strFiled == "TB" && strQuery == "5")
				{
					strQuery = "1";
				}
			}

			// ӽβͨ
			if (bNeedWild)
			{
				strQuery += "*";
			}
			else   // 
			{
				strQuery = "\"" + strQuery + "\"";
			}

			Xapian::Query subQuery;
			if (bAddFiled)
			{
				std::string strFiledTemp = iteCond->first;
				if (iteCond->first == kLinkPort)
				{
					strFiledTemp = kPassPort;
				}
				subQuery = oQueryParser.parse_query(strQuery,
					Xapian::QueryParser::FLAG_DEFAULT | Xapian::QueryParser::FLAG_WILDCARD,
					strFiledTemp);
			}
			else
			{
				subQuery = oQueryParser.parse_query(strQuery,
					Xapian::QueryParser::FLAG_DEFAULT | Xapian::QueryParser::FLAG_WILDCARD,
					iteCond->first);
			}

			if (iteStr == iteCond->second.begin())
			{
				andQuery = subQuery;
				DealFillCreateCondition(strFiled, strFillCreator, oQueryParser, andQuery);
			}
			else
			{
				if (bAddFiled)
				{
					andQuery = Xapian::Query(Xapian::Query::OP_AND, andQuery, subQuery);
				}
				else
				{
					andQuery = Xapian::Query(Xapian::Query::OP_OR, andQuery, subQuery);
				}

			}
		}
		if (iteCond == oCondMap.begin())
		{
			oQuery = andQuery;
		}
		else
		{
			oQuery = Xapian::Query(Xapian::Query::OP_AND, oQuery, andQuery);
		}
	}

	// Χ
	std::string strRange;
	int iCount = 1;
	std::map<std::string, RangeFieldValue>::const_iterator iterRangeCond;
	for (iterRangeCond = oRangeCondMap.begin();
		iterRangeCond != oRangeCondMap.end();
		++iterRangeCond, ++iCount)
	{
		const RangeFieldValue &stRange = iterRangeCond->second;
		if (!stRange.strStart.empty())
		{
			strRange += stRange.strStart + iterRangeCond->first;
			strRange += "..";
		}
		else
		{
			strRange += "..";
		}

		if (!stRange.strEnd.empty())
		{
			strRange += stRange.strEnd + iterRangeCond->first;
		}
		else
		{
			strRange += iterRangeCond->first;
		}

		if (iCount < (int)oRangeCondMap.size())
		{
			strRange += " AND ";
		}
	}
	if (!strRange.empty())
	{
		Xapian::Query rangeQuery = oQueryParser.parse_query(strRange);
		if (oQuery.empty())
		{
			oQuery = rangeQuery;
			SAFE_COUT << "ConstructQueryObjectTree end" << std::endl;
			return false;
		}

		oQuery = Xapian::Query(Xapian::Query::OP_AND, oQuery, rangeQuery);
	}

	SAFE_COUT << "ConstructQueryObjectTree end" << std::endl;
	return true;
}

//
void SetSortField(
	const std::vector< std::pair<std::string, bool> > &oSortRangeFieldVec,
	Xapian::Enquire &enquire,
	Xapian::MultiValueKeyMaker &oValueMaker)
{
	if (!oSortRangeFieldVec.empty())
	{
		std::vector< std::pair<std::string, bool> >::const_iterator iterSortField;
		bool bFind = false;
		for (iterSortField = oSortRangeFieldVec.begin();
			iterSortField != oSortRangeFieldVec.end();
			++iterSortField)
		{
			std::map<std::string, int>::iterator iteFind = gRangeField.find(iterSortField->first);
			if (iteFind != gRangeField.end())
			{
				bFind = true;
				oValueMaker.add_value(iteFind->second, iterSortField->second);
			}
		}

		if (bFind)
		{
#ifdef XAPIAN_1_4_25
			enquire.set_sort_by_key(&oValueMaker, false);
#else
			enquire.set_sort_by_key(&oValueMaker);
#endif
		}
	}
}

bool QueryBusinessID(
	const Xapian::Query &stQuery,
	const std::map<std::string, std::vector<std::string> > &mapCond,
	const std::vector<std::pair<std::string, bool> > &vecSortRangeField,
	const int &iOffSet, const int &iPageSize,
	std::vector<int> &vecBusinessiD,
	const std::string& strFillCreator = "")
{
	SAFE_COUT << "QueryBusinessID by Query begin" << std::endl;

	bool bRet = true;

	std::shared_lock<std::shared_mutex> readLock(gMutex);
	SAFE_COUT << "QueryBusinessID by Query readLock" << std::endl;

	if (!gWDataBase)
	{
		SAFE_COUT << "Null of Point to DB" << std::endl;
		return false;
	}

	try
	{
		Xapian::Database db(kDbPath);
		Xapian::Query stTempQuery = stQuery;

		if (!mapCond.empty())
		{
			Xapian::QueryParser oQueryParser;
			oQueryParser.set_database(db);

			PrepareQueryParser(oQueryParser);
			Xapian::Query oQuery;
			std::map<std::string, RangeFieldValue> mapRangeCond;
			bRet = ConstructQueryObjectTree(mapCond,
				mapRangeCond, oQueryParser, oQuery, strFillCreator);
			assert(bRet == true);
			if (!bRet)
			{
				SAFE_COUT << "QueryBusinessID by Query fail, ConstructQueryObjectTree error" << std::endl;
				return bRet;
			}

			if (stTempQuery.empty())
			{
				stTempQuery = oQuery;
			}
			else
			{
				stTempQuery = Xapian::Query(Xapian::Query::OP_AND, stTempQuery, oQuery);
			}
		}

		// Use an Enquire object on the database to run the query.
		Xapian::Enquire enquire(db);
		enquire.set_query(stTempQuery);
		// sort
		Xapian::MultiValueKeyMaker oValueMaker;
		SetSortField(vecSortRangeField, enquire, oValueMaker);

		// get result
		Xapian::MSet mset = enquire.get_mset(iOffSet, iPageSize + 1);

		for (Xapian::MSetIterator m = mset.begin(); m != mset.end(); ++m)
		{
			const std::string &data = m.get_document().get_data();
			// int iBusinessID = boost::lexical_cast<int>(data);
			int iBusinessID = std::stoi(data);
			vecBusinessiD.push_back(iBusinessID);
		}
	}
	catch (const Xapian::Error &ex)
	{
		SAFE_COUT << ex.get_msg() << std::endl;
	}

	SAFE_COUT << "QueryBusinessID by Query end" << std::endl;
	return bRet;
}

//
bool ConstructQuery(
	const std::map<std::string, std::vector<std::string>> &mapCond,
	const std::map<std::string, RangeFieldValue> &mapRangeCond,
	Xapian::Query &stQuery)
{
	SAFE_COUT << "ConstructQuery begin" << std::endl;
	bool bRet = true;

	std::shared_lock<std::shared_mutex> readLock(gMutex);
	SAFE_COUT << "ConstructQuery readLock" << std::endl;

	if (!gWDataBase)
	{
		SAFE_COUT << "Null of Point to DB" << std::endl;
		return false;
	}

	try
	{
		//Construct Query object tree
		Xapian::QueryParser oQueryParser;
		Xapian::Database db(kDbPath);
		oQueryParser.set_database(db);

		PrepareQueryParser(oQueryParser);

		bRet = ConstructQueryObjectTree(mapCond,
			mapRangeCond, oQueryParser, stQuery, "");
		assert(bRet == true);
		if (!bRet)
		{
			SAFE_COUT << "ConstructQuery fail, ConstructQueryObjectTree error" << std::endl;
			return bRet;
		}
	}
	catch (const Xapian::Error &ex)
	{
		SAFE_COUT << ex.get_msg() << std::endl;
	}

	SAFE_COUT << "ConstructQuery end" << std::endl;
	return bRet;
}


bool ConvertCondition(
	std::map<std::string, std::vector<std::string>>& relatedCondMap,
	std::map<std::string, std::vector<std::string>>& nonRelatedCondMap,
	std::map<std::string, RangeFieldValue>& rangeCondMap,
	Xapian::Query& relatedQuery,
	Xapian::Query& nonRelatedQuery)
{
	bool bRet = true;

	if (!nonRelatedCondMap.empty())
	{
		std::map<std::string, std::vector<std::string> > mapNonRelated;
		std::map<std::string, std::vector<std::string> >::iterator itFindCR = nonRelatedCondMap.find("CR");
		if (itFindCR != nonRelatedCondMap.end())
		{
			mapNonRelated["CR"] = itFindCR->second;
			nonRelatedCondMap.erase(itFindCR);
		}

		std::map<std::string, std::vector<std::string> >::iterator itFindCN = nonRelatedCondMap.find("CN");
		if (itFindCN != nonRelatedCondMap.end())
		{
			mapNonRelated["CN"] = itFindCN->second;
			nonRelatedCondMap.erase(itFindCN);
		}

		std::map<std::string, std::vector<std::string> >::iterator itFindProtect = nonRelatedCondMap.find("IPP");
		if (itFindProtect != nonRelatedCondMap.end())
		{
			mapNonRelated["IPP"] = itFindProtect->second;
			nonRelatedCondMap.erase(itFindProtect);
		}

		std::map<std::string, std::vector<std::string> >::iterator itFindFilter = nonRelatedCondMap.find("BF");
		if (itFindFilter != nonRelatedCondMap.end())
		{
			mapNonRelated["BF"] = itFindFilter->second;
			nonRelatedCondMap.erase(itFindFilter);
		}

		std::map<std::string, std::vector<std::string> >::iterator itFindNT = nonRelatedCondMap.find("NT");
		if (itFindNT != nonRelatedCondMap.end())
		{
			mapNonRelated["NT"] = itFindNT->second;
			nonRelatedCondMap.erase(itFindNT);
		}

		std::map<std::string, std::vector<std::string> >::iterator itFindTB = nonRelatedCondMap.find("TB");
		if (itFindTB != nonRelatedCondMap.end())
		{
			mapNonRelated["TB"] = itFindTB->second;
			nonRelatedCondMap.erase(itFindTB);
		}

		std::map<std::string, std::vector<std::string> >::iterator itFindPCA = nonRelatedCondMap.find("PCA");
		if (itFindPCA != nonRelatedCondMap.end())
		{
			mapNonRelated["PCA"] = itFindPCA->second;
			nonRelatedCondMap.erase(itFindPCA);
		}

		bRet = ConstructQuery(nonRelatedCondMap, rangeCondMap, nonRelatedQuery);
		assert(bRet == true);
		if (!bRet)
		{
			SAFE_COUT << "ConstructQuery failed" << std::endl;
			return bRet;
		}

		nonRelatedCondMap = mapNonRelated;
		rangeCondMap.clear();
	}
	else if (!rangeCondMap.empty())
	{
		Xapian::Query stTmpQuery;
		bRet = ConstructQuery(nonRelatedCondMap, rangeCondMap, stTmpQuery);
		assert(bRet == true);
		if (!bRet)
		{
			SAFE_COUT << "ConstructQuery failed." << std::endl;
			return bRet;
		}

		if (nonRelatedQuery.empty())
		{
			nonRelatedQuery = stTmpQuery;
		}
		else
		{
			nonRelatedQuery = Xapian::Query(Xapian::Query::OP_AND, nonRelatedQuery, stTmpQuery);
		}

		rangeCondMap.clear();
	}

	if (!relatedCondMap.empty())
	{
		bRet = ConstructQuery(relatedCondMap, rangeCondMap, relatedQuery);
		assert(bRet == true);
		if (!bRet)
		{
			SAFE_COUT << "ConstructQuery failed." << std::endl;
			return bRet;
		}

		relatedCondMap.clear();
	}

	return bRet;
}

bool RunRead(
	const std::map<std::string, std::vector<std::string>>& relatedCondMap,
	const std::map<std::string, std::vector<std::string>>& nonRelatedCondMap,
	const std::map<std::string, RangeFieldValue>& rangeCondMap,
	const std::vector<std::pair<std::string, bool>>& sortRangeField,
	const Xapian::Query& relatedQuery,
	const Xapian::Query& nonRelatedQuery)
{
	bool bNonRelated = (nonRelatedCondMap.empty() && nonRelatedQuery.get_length() == 0);
	bool bRelated = relatedQuery.get_length() == 0;

	bool bRet;

	int num = 10;

	//SAFE_COUT << "nonRelatedQuery type name: "
	//	<< typeid(nonRelatedQuery).name() << std::endl;

	//size_t subqueries_num = nonRelatedQuery.get_num_subqueries();
	//for (size_t i = 0; i < subqueries_num; i++)
	//{
	//	SAFE_COUT << "sub query[" << i << "]:"
	//		<< typeid(nonRelatedQuery.get_subquery(i)).name()
	//		<< std::endl;
	//}

	//while (num-- > 0)
	{
		//SAFE_COUT << "num: " << num << std::endl;

		std::vector<int> vecNonRelatedID;
		if (!bNonRelated)
		{
			bRet = QueryBusinessID(
				nonRelatedQuery,
				std::map<std::string, std::vector<std::string>>(),
				sortRangeField,
				0, 200000,
				vecNonRelatedID);
		}

		std::vector<int> vecRelatedID;
		if (!bRelated)
		{
			bRet = QueryBusinessID(
				relatedQuery,
				std::map<std::string, std::vector<std::string>>(),
				sortRangeField,
				0, 200000,
				vecRelatedID);
		}

		//std::this_thread::sleep_for(std::chrono::milliseconds(1));
	}

	return bRet;
}

class TestRead
{
public:
	TestRead() {}
	TestRead(const std::map<std::string, std::vector<std::string>>& relatedCondMap,
		const std::map<std::string, std::vector<std::string>>& nonRelatedCondMap,
		const std::map<std::string, RangeFieldValue>& rangeCondMap,
		const std::vector<std::pair<std::string, bool>>& sortRangeField)
		: mRelatedCondMap(relatedCondMap),
		mNonRelatedCondMap(nonRelatedCondMap),
		mRangeCondMap(rangeCondMap),
		mSortRangeField(sortRangeField)
	{}
	TestRead(const std::map<std::string, std::vector<std::string>>& relatedCondMap,
		const std::map<std::string, std::vector<std::string>>& nonRelatedCondMap,
		const std::map<std::string, RangeFieldValue>& rangeCondMap,
		const std::vector<std::pair<std::string, bool>>& sortRangeField,
		const Xapian::Query& relatedQuery,
		const Xapian::Query& nonRelatedQuery)
		: mRelatedCondMap(relatedCondMap),
		mNonRelatedCondMap(nonRelatedCondMap),
		mRangeCondMap(rangeCondMap),
		mSortRangeField(sortRangeField),
		mRelatedQuery(relatedQuery),
		mNonRelatedQuery(nonRelatedQuery),
		mQueryObjConstructed(true)
	{}
	~TestRead()	{}

	virtual bool Run()
	{
		Xapian::Query relatedQuery;
		Xapian::Query nonRelatedQuery;

		if (mQueryObjConstructed)
		{
			relatedQuery = mRelatedQuery;
			nonRelatedQuery = mNonRelatedQuery;
		}
		else
		{
			bool ok = false;
			if (!mNonRelatedCondMap.empty())
			{
				ok = ConstructQuery(mNonRelatedCondMap, mRangeCondMap, nonRelatedQuery);
				assert(ok == true);
			}
			else if (!mRangeCondMap.empty())
			{
				ok = ConstructQuery(mNonRelatedCondMap, mRangeCondMap, nonRelatedQuery);
				assert(ok == true);
			}

			if (!mRelatedCondMap.empty())
			{
				ok = ConstructQuery(mRelatedCondMap, mRangeCondMap, relatedQuery);
				assert(ok == true);
			}
		}

		bool bNonRelated = (mNonRelatedCondMap.empty() && nonRelatedQuery.get_length() == 0);
		bool bRelated = relatedQuery.get_length() == 0;

		bool bRet;

		std::vector<int> vecNonRelatedID;
		if (!bNonRelated)
		{
			bRet = QueryBusinessID(
				nonRelatedQuery,
				std::map<std::string, std::vector<std::string>>(),
				mSortRangeField,
				0, 200000,
				vecNonRelatedID);
		}

		std::vector<int> vecRelatedID;
		if (!bRelated)
		{
			bRet = QueryBusinessID(
				relatedQuery,
				std::map<std::string, std::vector<std::string>>(),
				mSortRangeField,
				0, 200000,
				vecRelatedID);
		}

		return bRet;
	}

private:
	std::map<std::string, std::vector<std::string>> mRelatedCondMap;
	std::map<std::string, std::vector<std::string>> mNonRelatedCondMap;
	std::map<std::string, RangeFieldValue> mRangeCondMap;
	std::vector<std::pair<std::string, bool>> mSortRangeField;

	Xapian::Query mRelatedQuery;
	Xapian::Query mNonRelatedQuery;
	bool mQueryObjConstructed = false;
};

//////////////////////////////////////////////////////////////////////////

bool ReadData(const std::string& query)
{
	try
	{
		Xapian::Database db(kDbPath);
		Xapian::Enquire enquire(db);

		Xapian::Query oQuery = Xapian::Query(query);
		enquire.set_query(oQuery);

		Xapian::MSet result = enquire.get_mset(0, db.get_doccount());

		// get all keys of 'query'
		std::vector<std::string> keys;
		std::map<std::string, std::vector<std::string>> terms; // <key, terms>
		for (Xapian::MSetIterator m = result.begin(); m != result.end(); ++m)
		{
			Xapian::Document doc = m.get_document();
			std::string key = doc.get_data();
			keys.push_back(key);

			for (Xapian::TermIterator t = doc.termlist_begin();	t != doc.termlist_end(); ++t)
			{
				std::vector<std::string>& term = terms[key];
				term.push_back(*t);
			}
		}
		std::sort(keys.begin(), keys.end());
		int count = result.get_matches_estimated();
	}
	catch (...)
	{
	}

	return true;
}

int GetRandom(int min, int max)
{
	std::mt19937 generator(std::random_device{}());
	std::uniform_int_distribution<int> distribution(min, max);
	return distribution(generator);
}

long long GetCurrentTimestamp()
{
	auto now = std::chrono::system_clock::now();
	auto nowC = std::chrono::system_clock::to_time_t(now);
	std::tm* nowTm = std::localtime(&nowC);
	auto nowMs = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch());
	return nowMs.count();
}

bool PrepareData(std::vector<XapianData>& data)
{
	// read data from text file
	// which collected by msg report in UNM like:
	/*
	std::ofstream outFile(kDataPath);
	if (!outFile)
	{
		// error
	}

	for (const auto& data, vecData)
	{
		// busiId
		outFile << "busiId:" << std::endl;
		outFile << data.m_iBusiId << std::endl;

		// fieldValue
		outFile << "fieldValue:" << std::endl;
		for (const auto& pair : data.m_mapFieldValue)
		{
			outFile << pair.first << "|" << pair.second << "\n";
		}

		// rangeFieldValue
		outFile << "rangeFieldValue:" << std::endl;
		for (const auto& pair : data.m_mapRangeFieldValue)
		{
			outFile << pair.first << "|" << pair.second << "\n";
		}

		outFile << std::endl;
	}
	outFile.close();
	*/

	std::ifstream inFile(kDataPath);
	if (!inFile)
	{
		return false;
	}

	std::string line;
	XapianData d;
	while (std::getline(inFile, line))
	{	
		if (line == "busiId:") // busiId
		{
			std::getline(inFile, line);
			d.busiId = std::stoi(line);
		}
		else if (line == "fieldValue:")
		{
			while (std::getline(inFile, line))
			{
				if (line == "rangeFieldValue:")
				{
					break;
				}

				// fieldValue
				std::istringstream iss(line);
				std::string key;
				std::string value = "";
				static std::string lastKey;
				if (line.find('|') == std::string::npos)
				{
					//NA | 1.S 0 2.X G E _ 1 / V P - 3 < -- > 2.S 0 2.X G E _ 1 / V P - 3 - 1 5 3 4 9 1 8 6 5 7
					//	1.S 0 2.X G E _ 1 / V P - 3 < -- > 2.S 0 2.X G E _ 1 / V P - 3 - 1 5 3 4 9 1 8 6 5 7
					//	1.S 0 2.X G E _ 1 / V P - 3 < -- > 2.S 0 2.X G E _ 1 / V P - 3 - 1 5 3 4 9 1 8 6 5 7
					if (d.fieldValue.find(lastKey) != d.fieldValue.end())
					{
						d.fieldValue[lastKey] = d.fieldValue[lastKey] + "\n" + line;
					}
				}
				else
				{
					if (std::getline(iss, key, '|'))
					{
						lastKey = key;
						if (key == "PCA")
						{
							int num = GetRandom(1, 81);
							d.fieldValue[key] = std::to_string(num);
						}
						else
						{
							std::getline(iss, value, '|');
							d.fieldValue[key] = value;
						}
					}
				}
			}

			// rangeFieldValue
			while (std::getline(inFile, line))
			{
				if (line.empty())
				{
					data.push_back(d);
					d.Clear();
					break;
				}

				std::istringstream iss(line);
				std::string key;
				std::string value = "";
				if (std::getline(iss, key, '|'))
				{
					if (key == "MT")
					{
						long long curTime = GetCurrentTimestamp();
						d.rangeFieldValue[key] = std::to_string(curTime);
					}
					else
					{
						std::getline(iss, value);
						d.rangeFieldValue[key] = value;
					}
				}
			}
		}
	}

	inFile.close();

	return true;
}

bool BuildIndexWithoutLock(
	const int &iBusinessID,
	const std::map<std::string, std::string> &oFieldValueMap,
	const std::map<std::string, std::string> &oRangeFieldValueMap)
{
	SAFE_COUT << "BuildIndex begin: " << iBusinessID << std::endl;

	std::string strBussinessID = std::to_string(iBusinessID);

	Xapian::TermGenerator oTermGenerator;
	Xapian::Document oDoc;

	try
	{
		oTermGenerator.set_document(oDoc);

		std::map<std::string, std::string>::const_iterator iterFieldValue;
		for (iterFieldValue = oFieldValueMap.begin();
			iterFieldValue != oFieldValueMap.end();
			++iterFieldValue)
		{
			oTermGenerator.index_text(iterFieldValue->second, 1, iterFieldValue->first);
			// full text search
			//oTermGenerator.index_text(iterFieldValue->second);
			oTermGenerator.increase_termpos();
		}
		oDoc.set_data(strBussinessID);

		std::map<std::string, std::string>::const_iterator iterRangeFieldValue;
		for (iterRangeFieldValue = oRangeFieldValueMap.begin();
			iterRangeFieldValue != oRangeFieldValueMap.end();
			++iterRangeFieldValue)
		{
			const std::string &strRange = iterRangeFieldValue->first;
			std::map<std::string, int>::const_iterator iteFind = gRangeField.find(strRange);
			if (iteFind != gRangeField.end())
			{
				// double dValue = boost::lexical_cast<double>(iterRangeFieldValue->second);
				double dValue = std::stod(iterRangeFieldValue->second);
				oDoc.add_value(iteFind->second, Xapian::sortable_serialise(dValue));
			}
			else
			{
				SAFE_COUT << "BuildIndex fail, Not Exist Range Condition" << std::endl;
				return false;
			}
		}

		std::string idterm = "KEY" + strBussinessID;
		oDoc.add_boolean_term(idterm);

		//if (!g_bMultiThreadInit)
		{
			std::unique_lock<std::shared_mutex> writeLock(gMutex);
			SAFE_COUT << "BuildIndex Lock" << std::endl;
			gWDataBase->replace_document(idterm, oDoc);
			++gBatchCount;

			if (gBatchCount > 1000)
			{
				gWDataBase->commit();
				gBatchCount = 0;
			}
		}
		//else
		//{
		//	XPWDataBasePtr pTmpWDataBase = GetTempDB();
		//	pTmpWDataBase->replace_document(idterm, oDoc);
		//}
	}
	catch (const Xapian::Error &ex)
	{
		SAFE_COUT << ex.get_msg() << std::endl;
	}

	SAFE_COUT << "BuildIndex end" << std::endl;
	return true;
}

bool BuildIndex(
	const int &iBusinessID,
	const std::map<std::string, std::string> &oFieldValueMap,
	const std::map<std::string, std::string> &oRangeFieldValueMap)
{
	return BuildIndexWithoutLock(iBusinessID, oFieldValueMap, oRangeFieldValueMap);
}

bool Commit()
{
	SAFE_COUT << "Commit begin" << std::endl;

	std::unique_lock<std::shared_mutex> writeLock(gMutex);
	SAFE_COUT << "Commit Lock" << std::endl;
	if (!gWDataBase)
	{
		SAFE_COUT << "Commit fail. gWDataBase is null" << std::endl;
		return false;
	}

	if (gBatchCount > 0)
	{
		gWDataBase->commit();
		gBatchCount = 0;
	}

	SAFE_COUT << "Commit end" << std::endl;
	return true;
}

bool RunWrite(const std::vector<XapianData>& data)
{
	int count = 0;
	const int BATCH_COUNT = 20;

	for (const auto& d : data)
	{
		BuildIndex(d.busiId, d.fieldValue, d.rangeFieldValue);
		++count;

		if (count % BATCH_COUNT == 0)
		{
			Commit();
		}
	}

	if (count % BATCH_COUNT != 0)
	{
		Commit();
	}

	return true;
}

class TestWrite
{
public:
	TestWrite() {}
	TestWrite(const std::vector<XapianData>& data) : mData(data) {}
	~TestWrite() {}

	virtual bool Run()
	{
		int count = 0;
		const int BATCH_COUNT = 20;

		for (const auto& d : mData)
		{
			BuildIndex(d.busiId, d.fieldValue, d.rangeFieldValue);
			++count;

			if (count % BATCH_COUNT == 0)
			{
				Commit();
			}
		}

		if (count % BATCH_COUNT != 0)
		{
			Commit();
		}

		return true;
	}

private:
	std::vector<XapianData> mData;
};

bool Run(int type = kTestTypeNone)
{
	bool testRead = (type & kTestTypeRead) != 0;
	bool testWrite = (type & kTestTypeWrite) != 0;

	//RunRead(gRelatedCondMap, gNonRelatedCondMap, gRangeCondMap, gSortRangeField,
	//	gRelatedQuery, gNonRelatedQuery);

	//for (int i = 0; i < 100; i++)
	//{
	//	SAFE_COUT << i << std::endl;
	//	RunRead(gRelatedCondMap, gNonRelatedCondMap, gRangeCondMap, gSortRangeField,
	//		gRelatedQuery, gNonRelatedQuery);
	//}

	const int kReadThreadCount = 100;
	std::vector<std::thread> tr;
	if (testRead)
	{
		for (int i = 0; i < kReadThreadCount; i++)
		{
#ifdef USE_GLOBAL_XAPIAN_QUERY_OBJECT
			std::shared_ptr<TestRead> r(new TestRead(gRelatedCondMap, 
				gNonRelatedCondMap,	gRangeCondMap, gSortRangeField, 
				gRelatedQuery, gNonRelatedQuery));
#else
			std::shared_ptr<TestRead> r(new TestRead(gRelatedCondMap, 
				gNonRelatedCondMap, gRangeCondMap, gSortRangeField));
#endif // USE_GLOBAL_XAPIAN_QUERY_OBJECT
			//tr.emplace_back(std::bind(&TestRead::Run, r));
			tr.emplace_back([r]() {	r->Run(); });
		}
	}

	const int kWriteThreadCount = 3;
	std::vector<std::thread> tw;
	if (testWrite)
	{
		for (int i = 0; i < kWriteThreadCount; i++)
		{
			std::shared_ptr<TestWrite> w(new TestWrite(gData));
			tw.emplace_back([w]() {	w->Run(); });
		}
	}

	if (testRead)
	{
		for (auto& t : tr)
		{
			if (t.joinable())
			{
				t.join();
			}
		}
	}

	if (testWrite)
	{
		for (auto& t : tw)
		{
			if (t.joinable())
			{
				t.join();
			}
		}
	}

	return true;
}

//////////////////////////////////////////////////////////////////////////
void UnmTest()
{
	// Specify the type to test
	int type = kTestTypeRead /*| kTestTypeWrite*/;

#ifdef SPECIFY_TYPE_IN_STD_INPUT
	SAFE_COUT << "Specify the test type(1-Read 2-Write 3-Read&Write): ";
	while (!(std::cin >> type) || type < 1 || type > 3)
	{
		std::cin.clear();
#pragma push_macro("max")
#undef max
		std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
#pragma pop_macro("max")

		SAFE_COUT << "Invalid input, pls try again: ";
	}
#endif

	if (type == kTestTypeNone)
	{
		SAFE_COUT << "Nothing to test!" << std::endl;
		return;
	}

	//////////////////////////////////////////////////////////////////////////

	if (!gWDataBase)
	{
		gWDataBase = XPWDataBasePtr(new Xapian::WritableDatabase(
			kDbPath, Xapian::DB_CREATE_OR_OPEN));
	}

	//////////////////////////////////////////////////////////////////////////
	// Prepare read data

	// m_oRangeField
	std::vector<std::string>::const_iterator iteRange = gRangeFieldNameVec.begin();
	int slot = 1;
	for (; iteRange != gRangeFieldNameVec.end(); ++iteRange)
	{
		gRangeField[*iteRange] = slot++;
	}
	std::sort(gRangeFieldNameVec.begin(), gRangeFieldNameVec.end());

	gNonRelatedCondMap["SL"] = std::vector<std::string>(1, "tunnel");
	gNonRelatedCondMap["TB"] = std::vector<std::string>(1, "1");

	RangeFieldValue rangeField;
	gRangeCondMap["LR"] = rangeField;

	gSortRangeField.push_back(std::make_pair("MT", true));

	// Construct global Xapian::Query object
	ConvertCondition(gRelatedCondMap, gNonRelatedCondMap, gRangeCondMap,
		gRelatedQuery, gNonRelatedQuery);

	//if (!gNonRelatedCondMap.empty())
	//{
	//	ConstructQuery(gNonRelatedCondMap, gRangeCondMap, gNonRelatedQuery);
	//}
	//else if (!gRangeCondMap.empty())
	//{
	//	ConstructQuery(gNonRelatedCondMap, gRangeCondMap, gNonRelatedQuery);
	//}

	//if (!gRelatedCondMap.empty())
	//{
	//	ConstructQuery(gRelatedCondMap, gRangeCondMap, gRelatedQuery);
	//}

	//////////////////////////////////////////////////////////////////////////
	// Prepare write data
	// ReadData("SLtunnel");
	PrepareData(gData);

	//////////////////////////////////////////////////////////////////////////

	int count = 0;
	while (true)
	{
		SAFE_COUT << "============ Running: " << ++count << " =============" << std::endl;
		Run(type);
		std::this_thread::sleep_for(std::chrono::milliseconds(100));
	}
}