/*
c++ -std=c++1z -pedantic -Wall -Wextra -O3 -lbenchmark -lbenchmark_main -I. -I./src -I../src -I.. -o benchmark-encode_length.2 ./benchmark-encode_length.2.cc && ./benchmark-encode_length.2
*/
#include <limits>

#include "benchmark/benchmark.h"


template<class T>
std::string
encode_length_original(T len)
{
	std::string result;
	if (len < 255) {
		result += static_cast<unsigned char>(len);
	} else {
		result += '\xff';
		len -= 255;
		while (true) {
			unsigned char b = static_cast<unsigned char>(len & 0x7f);
			len >>= 7;
			if (!len) {
				result += char(b | static_cast<unsigned char>(0x80));
				break;
			}
			result += b;
		}
	}
	return result;
}

const unsigned long long x = 0xff;
//const unsigned long long x = std::numeric_limits<unsigned long>::max() - 10;
static void BM_EncodeLength_Original(benchmark::State& state) {
	for (auto _ : state) {
		std::string s;
		for (auto i = x; i != 0; i >>= 1) {
			s += encode_length_original(i);
		}
	}
}
BENCHMARK(BM_EncodeLength_Original);


template<class T>
std::string
encode_length_optimized(T len)
{
	std::string result;
	if (len < 255) {
		result += static_cast<unsigned char>(len);
		return result;
	} else {
		result += '\xff';
		len -= 255;
		while (true) {
			unsigned char b = static_cast<unsigned char>(len & 0x7f);
			len >>= 7;
			if (!len) {
				result += b | static_cast<unsigned char>(0x80);
				break;
			}
			result += b;
		}
		return result;
	}
}
static void BM_EncodeLength_Optimized(benchmark::State& state) {
	for (auto _ : state) {
		std::string s;
		for (auto i = x; i != 0; i >>= 1) {
			s += encode_length_optimized(i);
		}
	}
}
BENCHMARK(BM_EncodeLength_Optimized);


template<class U>
inline void
pack_uint(std::string & s, U value)
{
	static_assert(std::is_unsigned<U>::value, "Unsigned type required");

	while (value >= 128) {
		s += static_cast<char>(static_cast<unsigned char>(value) | 0x80);
		value >>= 7;
	}
	s += static_cast<char>(value);
}
static void BM_EncodeLength_Pack(benchmark::State& state) {
	for (auto _ : state) {
		std::string s;
		for (auto i = x; i != 0; i >>= 1) {
			pack_uint(s, i);
		}
	}
}
BENCHMARK(BM_EncodeLength_Pack);


inline void
pack_char(std::string & s, char value)
{
	s += value;
}
static void BM_EncodeLength_Char(benchmark::State& state) {
	for (auto _ : state) {
		std::string s;
		for (auto i = x; i != 0; i >>= 1) {
			pack_char(s, i);
		}
	}
}
BENCHMARK(BM_EncodeLength_Char);
