#!/usr/bin/env python

"""
setup.py file for Xapian HunSpell stemmer
"""

__version__ = '0.1'
__author__ = 'Evgeny Sizikov'
DESCRIPTION = """Xapian uses algorithmic Snowball stemmer by default.
However, there is a proven spelling solution used by OpenOffice.org
among others - Hunspell (https://hunspell.sf.net/. This project is to
develop a Hunspell-powered stemmer to be used with Xapian."""

from os import environ
XAPIAN_CONFIG = environ.get('XAPIAN_CONFIG', 'xapian-config')

from distutils.core import setup, Extension
import commands

# pkg-config support got from http://code.activestate.com/recipes/502261/
def pkgconfig(*packages, **kw):
    flag_map = {'-I': 'include_dirs', '-L': 'library_dirs', '-l': 'libraries'}
    for token in commands.getoutput('pkg-config --libs --cflags %s' % ' '.join(packages)).split():
        kw.setdefault(flag_map.get(token[:2]), []).append(token[2:])
    return kw

def config(prog, args='--libs --cflags', **kw):
    flag_map = {'-I': 'include_dirs', '-L': 'library_dirs', '-l': 'libraries'}
    for token in commands.getoutput('%s-config %s' % (prog, args)).split():
        kw.setdefault(flag_map.get(token[:2]), []).append(token[2:])
    return kw

def config_xapian(**kw):
    return config(XAPIAN_CONFIG.split('-')[0], '--libs --cxxflags', **kw)

xapian_hunspell_stemmer_module = Extension('_xapian_hunspell_stemmer',
                                           sources=['xapian_hunspell_stemmer.i',
                                                    'xapian_hunspell_stemmer.cpp'],
                                           swig_opts='-c++ -threads -shadow -modern -O -DXAPIAN_VISIBILITY_DEFAULT'.split() + \
                                                     commands.getoutput('%s --swigflags' % XAPIAN_CONFIG).split(),
                                           define_macros=[('CONVERT_HUNSPELL_DIC', None), ('HAVE_PATCHED_XAPIAN', None), ('CONVERT_USING_PYTHON_C_API', None)],
                                           undef_macros=['HAVE_PATCHED_XAPIAN', 'CONVERT_USING_PYTHON_C_API'],
                                           extra_compile_args=[],
                                           extra_link_args=[],
                                           **pkgconfig('hunspell', **config_xapian())
                                           )

setup(name = 'xapian_hunspell_stemmer',
      version = __version__,
      author = __author__,
      author_email = 'esizikov@gmail.com',
      license = 'GNU GPL',
      description = 'Hunspell stemmer for Xapian',
      long_description = DESCRIPTION,
      ext_modules = [xapian_hunspell_stemmer_module],
      py_modules = ['xapian_hunspell_stemmer'],
      )
